/* Copyright (c) 2004, 2009, Oracle and/or its affiliates. 
All rights reserved. */

/*
 * NAME
 * Demoagent1.cpp: Demonstrates agent to monitor a file
 *
 * DESCRIPTION
 * This is a sample program that demonstrates an agent to monitor
 * a file. The agent has the following tasks:
 *  - On startup         : Create the file.
 *  - On shutdown        : Delete the file.
 *  - On check command   : Detect whether the file is present or not.
 *  - On clean command   : Delete the file.
 * This program can be used as a basis to build up more complicated
 * agents that manage real-world resources.
 *
 * RELATED DOCUMENTS
 *  (1) readme.txt, which should be available in the same directory as this file.
 *  (2) demoagent2.cpp, a hybrid agent which monitors a file.
 *  (3) macrodemo.cpp, which demonstrates the use of the automated test scripts.
 *  (4) The Oracle Clusterware Administration and Deployment Guide - 11g Release 2
 *  (5) The Oracle Technology Network - available online at http://www.oracle.com/technology
 *
 * NOTES
 *  (1) The clsagfw_log messages are sent by default to the agent log directory. This 
 *      would typically be $(INSTALL_DIR)/log/<hostname>/agent/crsd/
 *  (2) The file to be monitored is determined by reading the PATH_NAME attribute for the
 *      resource. This attribute is set when the resource is defined. Look at the readme 
 *      file for more information.
 *  
 * MODIFIED
 *    skakarla   05/20/09 - adding log level
 *    vmanivel   03/25/09 - Updating based on comments
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <clsagfw.h>
#include <clsagfwm.h>
#include <oratypes.h>

#define TEST_TYPE1                   (oratext *) "test_type1"
#define TEST_PATHNAME_ATTR           (oratext *) "PATH_NAME"


/*
 * NAME 
 *    test_agent_exit
 * DESCRIPTION
 *    Function to clean up, called by framework whenever it receives 
 * the AGENT_EXIT message from crsd.
 * PARAMS
 *    exitCode, which is an sb4 describing the reason for exit.
 * RETURNS
 *    void
 */
void 
test_agent_exit(sb4 exitCode)
{
  clsagfw_log(NULL, 1, (oratext *)"Demo Agent is exiting..\n");
}


/*
 * NAME
 *    type1_start
 * DESCRIPTION
 *    Action entry point for the 'start' command. Creates the file for monitoring.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which could be CLSAGFW_AE_FAIL or CLSAGFW_AE_SUCCESS depending on whether
 *    the action failed or succeeded.
 */
ub4 
type1_start(const clsagfw_aectx *ectx)
{
    ub4  ret = CLSAGFW_AE_FAIL;
    const oratext   *pResName  = NULL;
    const oratext   *pPathName = NULL;
    FILE *fp;

    clsagfw_log(ectx, 1, (oratext *)"Start action called..");

    /* Try to read the resource name */
    if (clsagfw_get_attrvalue(ectx, (oratext *)"NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        goto done;
    }

    /* Try to read the PATH_NAME attribute */
    if (clsagfw_get_attrvalue(ectx, TEST_PATHNAME_ATTR, &pPathName) !=
            CLSAGFW_SUCCESS)
      {
	goto done;
      }

    clsagfw_log(ectx, 1, 
                (oratext *)"Start action arguments: resName = %s, pathName = %s", 
                pResName, pPathName);
    
    /* Try to create the file */
    fp = fopen((char *)pPathName, "w");

    if(!fp)
    {
      /* Could not open the file */
        clsagfw_log(ectx, 1, (oratext *) "START action for resource %s: FAILED\n", pResName);
        ret = CLSAGFW_AE_FAIL;
    }
    else
    {
      /* Opened the file successfully */
        clsagfw_log(ectx, 1, (oratext *) "START action for resource %s: SUCCESS\n", pResName);
        ret = CLSAGFW_AE_SUCCESS;
	fclose(fp);
    }

done:
    return ret;
}

/*
 * NAME
 *    type1_stop
 * DESCRIPTION
 *    Action entry point for the 'stop' and 'clean' commands. Deletes the file being monitored.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which could be CLSAGFW_AE_FAIL or CLSAGFW_AE_SUCCESS depending on whether
 *    the action failed or succeeded.
 */
ub4 
type1_stop (const clsagfw_aectx *ectx)
{
    ub4             ret = CLSAGFW_AE_FAIL;
    const oratext   *pResName  = NULL;
    const oratext   *pPathName = NULL;

    clsagfw_log(ectx, 1, (oratext *)"Stop action called..");

    /* Try to read the resource name */
    if (clsagfw_get_attrvalue(ectx, (oratext *)"NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        clsagfw_log(ectx, 1, (oratext *)"STOP action: %s: Could not read attribute\n", pResName);
	goto done;
    }

    /* Try to read the PATH_NAME attribute */
    if (clsagfw_get_attrvalue(ectx, TEST_PATHNAME_ATTR, &pPathName) !=
            CLSAGFW_SUCCESS)
    {
        pPathName = pResName;
    }

    clsagfw_log(ectx, 1, 
                (oratext *)"Stop action arguments: resName = %s, pathName = %s",
                pResName, pPathName);

    /* Try to delete the file */
    if (remove((char *)pPathName))
    {
      /* Could not delete the file */
        clsagfw_log(ectx, 1, (oratext *)"STOP action for resource %s: FAILED\n", pResName);
        ret = CLSAGFW_AE_FAIL;
    }
    else
    {
      /* Deleted the file successfully */
        clsagfw_log(ectx, 1, (oratext *)"STOP action for resource %s: SUCCESS\n", pResName);
        ret = CLSAGFW_AE_SUCCESS;
    }

done:
    return ret;
}

/*
 * NAME
 *    type1_check
 * DESCRIPTION
 *    Action entry point for the 'check' command. Determines if the file exists.
 * PARAMS
 *    Pointer to a clsagfw struct. This is created and deleted internally by the
 *    framework and is opaque to the agent. Do not modify it.
 * RETURNS
 *    ub4 which gives the status of the resource. Look at clsagfw.h for details on
 *    this function's return codes.
 */
ub4 
type1_check(const clsagfw_aectx *ectx)
{
    ub4             ret = CLSAGFW_UNKNOWN;
    const oratext   *pResName  = NULL;
    const oratext   *pPathName = NULL;
    FILE *fp;

    clsagfw_log(ectx, 1, (oratext *)"Check action called..");

    /* Try to read the resource name */
    if (clsagfw_get_attrvalue(ectx, (oratext *) "NAME", &pResName) != 
            CLSAGFW_SUCCESS)
    {
        goto done;
    }

    /* Try to read the PATH_NAME attribute */
    if (clsagfw_get_attrvalue(ectx, TEST_PATHNAME_ATTR, &pPathName) !=
            CLSAGFW_SUCCESS)
    {
        clsagfw_log(ectx, 1, (oratext *)"CHECK action: %s: Could not read attribute\n", pResName);
	goto done;
    }

    clsagfw_log(ectx, 1, 
                (oratext *)"Check action arguments: resName = %s, pathName = %s",
                pResName, pPathName);

    /* Check if the file is accessible */
    fp = fopen((char *)pPathName, "r");
    if (!fp)
    {
      /* Could not open file */
        clsagfw_log(ectx, 1, (oratext *)"CHECK action: %s status - UNPLANNED_OFFLINE\n", pResName);
        ret = CLSAGFW_UNPLANNED_OFFLINE;
    }
    else
    {
      /* Opened file successfully */
        clsagfw_log(ectx, 1,(oratext *)"CHECK action: %s status - ONLINE\n", pResName);
	fclose(fp);
        ret = CLSAGFW_ONLINE;
    }

done:
    return ret;
}

/*
 * Initialization of the agent framework and registration of types is done in main.
 */
int main(sb4 argc, oratext **argv)
{
    clsagfw_log(NULL, 1, (oratext *)" *** Agent Framework Demo Agent Started *** \n");

    /*
     * Initialize the agent framework 
     */ 
    if (clsagfw_init(argc, argv, 0, NULL, 0) 
        != CLSAGFW_SUCCESS)
    {
        clsagfw_log(NULL, 1, (oratext *)"Failed to initilize the agent framework\n");
        clsagfw_exit(-1);
    }

    /*
     * Set the exit callback function
     */
    clsagfw_set_exitcb(test_agent_exit);

    /*
     * Add the type definition to the framework
     */ 
    if (clsagfw_add_type(TEST_TYPE1) != CLSAGFW_SUCCESS)
    {
        clsagfw_log(NULL, 1,(oratext *)"Failed in adding type %s to the framework\n", TEST_TYPE1);
        clsagfw_exit(-1);
    }
    

    /*
     * Set all call back action entries for TEST_TYPE1
     */
    clsagfw_set_entrypoint(TEST_TYPE1, type1_start, CLSAGFW_ACTION_START);
    clsagfw_set_entrypoint(TEST_TYPE1, type1_stop, CLSAGFW_ACTION_STOP);
    clsagfw_set_entrypoint(TEST_TYPE1, type1_check, CLSAGFW_ACTION_CHECK);
    clsagfw_set_entrypoint(TEST_TYPE1, type1_stop, CLSAGFW_ACTION_CLEAN);

    clsagfw_log(NULL, 1, (oratext *)"Added resource type [%s] to the agent framework\n", TEST_TYPE1);

    /*
     * All set to go, Start the framework.
     * This function is a blocking call, it would return under only
     * one condition, that is when ever CRSD sends an AGENT_EXIT
     * command
     */
    clsagfw_startup();

    /*** NOT REACHED **/

    return 0; 
}


